﻿/*
    http://docs.phonegap.com/en/edge/cordova_file_file.md.html#File
    http://api.jquery.com/category/deferred-object/

    initialization (once):
        fs = new ONTIME.Fs()
        fs.init() -> (done)

    reading:
        fs.getFileEntry(path, flags) -> fileEntry
        fs.getFile(fileEntry) -> file
        reader = new ONTIME.FsReader();
        reader.readAsText(file) -> data

    reading/writing simplified/high level:
        ONTIME.Fs.readFile(path) -> (data)
        ONTIME.Fs.writeFile(path, data) -> (done)

    writing
        fs.getFileEntry(path, flags) -> fileEntry
        fs.createWriter(fileEntry) -> writer
        writer.write(data) -> (done)

    TODO: 
        implement middle level API:
            file = createFile(path)
            file.read() -> data
            file.write(data) -> (done)

    TODO:
        implement layered model:
            high: Abstract data API
            low: LocalStorage, PhoneGap/File

    TODO:
        implement instant persistance: PhoneGap/File, otherwise LocalStorage
*/

// ONTIME namespace 
var ONTIME = ONTIME || {};


ONTIME.Fs = function () {
    this.fs = null;
};

ONTIME.Fs.log = function (text) {
    console.log("ONTIME.Fs: " + text);
}

ONTIME.Fs.fail = function (dfd, msg, error) {
    ONTIME.Fs.log("[" + (typeof error === "object"?error.code:error) + "]" + msg);
    dfd.reject("[" + (typeof error === "object" ? error.code : error) + "]" + msg);
};

//-----------------------------------------------------------------
ONTIME.Fs.prototype.init = function () {
    this.fs = null;

    var dfd = new jQuery.Deferred();

    var fsPers = 1; // LocalFileSystem.PERSISTENT

    // default quota = 5MB
    //window.initPersistentFileSystem() 

    if (typeof window.requestFileSystem === "function") {
        window.requestFileSystem(1, 0,
            $.proxy(function (dfd, fileSystem) {
                this.fs = fileSystem;
                dfd.resolve();
            }, this, dfd),
            $.proxy(ONTIME.Fs.fail, this, dfd, "requestFileSystem failed")
        );
    } else {
        ONTIME.Fs.fail(dfd, "requestFileSystem is not available", {code: 99});
    }
    return dfd.promise();
};

//-----------------------------------------------------------------
ONTIME.Fs.prototype.resolveLocalFileSystemURL = function (path) {
    var dfd = new jQuery.Deferred();

    window.resolveLocalFileSystemURL(
        path,
        $.proxy(function (dfd, fileEntry) {
            dfd.resolve(fileEntry);
        }, this, dfd),
        $.proxy(ONTIME.Fs.fail, this, dfd, "resolveLocalFileSystemURL failed (" + path + ")")
    );

    return dfd.promise();
};

ONTIME.Fs.prototype.getFileEntry = function (path, flags) {
    var dfd = new jQuery.Deferred();

    //this.fs.root.getFile(path, flags,
    //    $.proxy(function (dfd, fileEntry) {
    //        dfd.resolve(fileEntry);
    //    }, this, dfd),
    //    $.proxy(ONTIME.Fs.fail, this, dfd, "root.getFile failed"));

    return dfd.promise();
};



//-----------------------------------------------------------------
ONTIME.Fs.prototype.createWriter = function (fileEntry) {
    var dfd = new jQuery.Deferred();
    fileEntry.createWriter(
        $.proxy(function (dfd, writer) {
            dfd.resolve(new ONTIME.FsWriter(writer));
        }, this, dfd),
        $.proxy(ONTIME.Fs.fail, this, dfd, "createWrite failed")
    );

    return dfd.promise();
};



ONTIME.FsWriter = function (nativeObj) {
    this.nativeObj = nativeObj;
};
ONTIME.FsWriter.prototype.write = function (text) {
    var dfd = new jQuery.Deferred();

    this.nativeObj.onwrite = $.proxy(function (dfd) {
        dfd.resolve();
    }, this, dfd);
    this.nativeObj.onerror = $.proxy(function (dfd, error) {
        dfd.reject(error);
    }, this, dfd);

    this.nativeObj.write(text);
    return dfd.promise();
};

ONTIME.FsReader = function () {
    this.nativeObj = new FileReader();
};
ONTIME.FsReader.prototype.readAsText = function (file) {
    var dfd = new jQuery.Deferred();

    this.nativeObj.onload = $.proxy(function (dfd, evt) {
        dfd.resolve(evt.target.result);
    }, this, dfd);
    this.nativeObj.onerror = $.proxy(ONTIME.Fs.fail, this, dfd, "readAsText failed (FileReader.onerror)");

    this.nativeObj.readAsText(file);
    return dfd.promise();
};


// simplified 
ONTIME.Fs.readFile = function (folder, path) {
    var dfd = new jQuery.Deferred();

    var fs = new ONTIME.Fs();
    fs.init()
    .done(function () {
        fs.resolveLocalFileSystemURL(folder + path)
        .done(function (fileEntry) {
            fileEntry.file(
                function (file) {
                    (new ONTIME.FsReader())
                    .readAsText(file)
                    .done(function (data) {
                        dfd.resolve(data);
                    })
                    .fail(function (error) {
                        ONTIME.Fs.log("readAsText failed");
                        dfd.reject(error);
                    });
                },
                function (error) {
                    ONTIME.Fs.log("getFile failed");
                    dfd.reject(error);
                }
            );
        })
        .fail(function (error) {
            dfd.reject(error);
        });
    })
    .fail(function (error) {
        ONTIME.Fs.log("init failed");
        dfd.reject(error);
    });

    return dfd.promise();
};

ONTIME.Fs.writeFile = function (folder, path, data) {
    var dfd = new jQuery.Deferred();

    var fs = new ONTIME.Fs();
    fs.init()
    .done(function () {
        fs.resolveLocalFileSystemURL(folder)
        .done(function (folderEntry) {
                folderEntry.getFile(
                    path, { create: true, exclusive: false },
                    function (fileEntry) {
                        fs.createWriter(fileEntry)
                            .done(function (writer) {
                                writer.write(data)
                                    .done(function () {
                                        dfd.resolve();
                                    })
                                    .fail(function (error) {
                                        dfd.reject(error)
                                    });
                            })
                            .fail(function (error) {
                                dfd.reject(error)
                            });
                    },
                    function (error) {
                        dfd.reject(error)
                    }
                );
        })
        .fail(function (error) {
            dfd.reject(error)
        });
    })
    .fail(function (error) {
        dfd.reject(error)
    });

    return dfd.promise();
};